/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id: User_win32.c $:

  Description:
    USER implemented functions called by the netXTransport Toolkit

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2021-01-21  - Added user thread function for cyclic netXTransport toolkit handling
    2020-11-25  - Removed dead code previously used for log-file handling
    2019-12-20  - Changed USER_TRACE to only create debugview outputs
    2019-11-01  - added USER_GetVersionString()
                - lint warnings removed
    2013-02-23  initial version

**************************************************************************************/

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>

#include "netXTransport.h"

#include "OS_Dependent.h"

/* Helpers for string concatination */
#define STR_EXPAND(tok) #tok
#define STR(tok) STR_EXPAND(tok)


typedef struct NXT_LOG_FILE_Ttag
{
  FILE* hFile;
} NXT_LOG_FILE_T;


/* Supply driver version strings in Trace and in User function */
static const char s_abVer[] = "NXT-Toolkit V"STR(NXT_VERSION_MAJOR)"."STR(NXT_VERSION_MINOR)"."STR(NXT_VERSION_BUILD)"."STR(NXT_VERSION_REV);


/*****************************************************************************/
/*! Print a trace message from netXtransport toolkit
*     \param ptDevInstance  Device instance the trace is coming from
*     \param ulTraceLevel   see TRACE_LVL_XXX defines
*     \param szFormat       printf style format string
*     \param ...            printf arguments                                 */
/*****************************************************************************/
void USER_Trace(NETX_TRANSPORT_DATA_T* ptnetXTransportInst, uint32_t ulTraceLevel, char* szFormat, ...)
{
  va_list         vaformat;
  char            szVarstring[NXT_MAX_PATH];
  char            szBuffer[NXT_MAX_PATH];
  SYSTEMTIME      SystemTime;

  UNREFERENCED_PARAMETER(ptnetXTransportInst);
  UNREFERENCED_PARAMETER(ulTraceLevel);

  GetLocalTime( &SystemTime);

  /* Don't print the date information to log */
  (void)sprintf_s( szBuffer, NXT_MAX_PATH,"NXTTKIT_: %.2d:%.2d:%.2d.%.3d: ",
                   SystemTime.wHour,
                   SystemTime.wMinute,
                   SystemTime.wSecond,
                   SystemTime.wMilliseconds);

  va_start( vaformat, szFormat);

  (void)vsnprintf_s( szVarstring, NXT_MAX_PATH, NXT_MAX_PATH-strlen(szBuffer),szFormat, vaformat);
  va_end( vaformat);

  (void)strcat_s(szBuffer, NXT_MAX_PATH, szVarstring);

  /* Use Debugprint() for the output */
  szBuffer[strlen(szBuffer)+1] = '\0';
  szBuffer[strlen(szBuffer)] = '\n';

  OutputDebugString( szBuffer);
} /*lint !e438 : last value assigned to 'vaformat' not used */

/*****************************************************************************/
/*! Initialize trace
*     \param ptnetXTransportInst Pointer to netXTransport instance           */
/*****************************************************************************/
void USER_TraceInitialize(NETX_TRANSPORT_DATA_T* ptnetXTransportInst)
{
  /* log file header */
  USER_Trace(ptnetXTransportInst, 0xFF, "----- netXTransport Log started ---------------------\n");
  USER_Trace(ptnetXTransportInst, 0xFF, "%s\n", s_abVer);
}

/*****************************************************************************/
/*! De-initialize trace
*     \param ptnetXTransportInst Pointer to netXTransport instance           */
/*****************************************************************************/
void USER_TraceDeInitialize(NETX_TRANSPORT_DATA_T* ptnetXTransportInst)
{

  UNREFERENCED_PARAMETER( ptnetXTransportInst);
}

/*****************************************************************************/
/*! Function retrieves connector specific timeout
*     \param ptConnector  Pointer to connector
*     \param ptConnector  Pointer to returned timeout                        */
/*****************************************************************************/
void USER_GetConnectorTimeout( PNETX_CONNECTOR_T ptConnector, uint32_t* pulTimeout)
{
  if (pulTimeout == NULL)
    return;

  UNREFERENCED_PARAMETER( ptConnector);

  /* currently set to default timeout */
  *pulTimeout = 100;
}

/*****************************************************************************/
/*! User defined send thread function (starts HilTransport Send-Handler)
*     \param lpParameter Pointer to win32 dependent thread information structure
*     returns 0                                                              */
/*****************************************************************************/
DWORD WINAPI USERSendThread( LPVOID lpParameter)
{
  WIN32_THREAD_PARAM_T* ptThreadParam = (WIN32_THREAD_PARAM_T*)lpParameter;

  if (NULL != ptThreadParam->pfnThreadFunction)
  {
    ptThreadParam->pfnThreadFunction( ptThreadParam->pvUserParam);
  }
  return 0;
}

/*****************************************************************************/
/*! User defined timeout thread function (starts HilTransport Timeout-Handler)
*     \param lpParameter Pointer to win32 dependent thread information structure
*     returns 0                                                              */
/*****************************************************************************/
DWORD WINAPI USERTimeoutThread( LPVOID lpParameter)
{
  WIN32_THREAD_PARAM_T* ptThreadParam = (WIN32_THREAD_PARAM_T*)lpParameter;

  if (NULL != ptThreadParam->pfnThreadFunction)
  {
    ptThreadParam->pfnThreadFunction( ptThreadParam->pvUserParam);
  }
  return 0;
}

/*****************************************************************************/
/*! Function retrieves OS specific thread parameter
*     \param szThreadName     Name of thread to be started
*     \param ppvThreadParam   Returned pointer to thread parameter
*     \param pulParamSize     Size of buffer pointed by *ppvThreadParam
*     \param pfThreadFunction Pointer to returned timeout parameter info
*     return NXT_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t USER_GetThreadParam( char* szThreadName, void** ppvThreadParam, uint32_t* pulParamSize, PFN_THREAD_FUNCTION pfThreadFunction, void* pvParam)
{
  WIN32_THREAD_PARAM_T* ptThreadParam = NULL;

  if (szThreadName == NULL)
    return -1;
  else if (0 == strncmp(szThreadName, "HilTransportSendThread", strlen(szThreadName)))
  {
    ptThreadParam = (WIN32_THREAD_PARAM_T*)OS_Memalloc(sizeof(WIN32_THREAD_PARAM_T));

    ptThreadParam->lptSecAtt         = NULL;
    ptThreadParam->tStackSize        = 0;
    ptThreadParam->lpStartAddr       = USERSendThread;
    ptThreadParam->pvParam           = ptThreadParam;
    ptThreadParam->dwFlags           = 0;
    ptThreadParam->lpID              = 0;
    ptThreadParam->pfnThreadFunction = pfThreadFunction;
    ptThreadParam->pvUserParam       = pvParam;

  } else if (0 == strncmp(szThreadName, "HilTransportTimeoutThread", strlen(szThreadName)))
  {
    ptThreadParam = (WIN32_THREAD_PARAM_T*)OS_Memalloc(sizeof(WIN32_THREAD_PARAM_T));

    ptThreadParam->lptSecAtt         = NULL;
    ptThreadParam->tStackSize        = 0;
    ptThreadParam->lpStartAddr       = USERTimeoutThread;
    ptThreadParam->pvParam           = ptThreadParam;
    ptThreadParam->dwFlags           = 0;
    ptThreadParam->lpID              = 0;
    ptThreadParam->pfnThreadFunction = pfThreadFunction;
    ptThreadParam->pvUserParam       = pvParam;
  }

  if (ppvThreadParam)
    *ppvThreadParam = ptThreadParam;

  if (pulParamSize)
    *pulParamSize = sizeof(*ptThreadParam);

  return 0;
}

/*****************************************************************************/
/*! Function frees user thread prameter (previously allocated with with USER_GetThreadParam())
*     \param ptConnector  Pointer to connector
*     \param ptConnector  Pointer to returned timeout                        */
/*****************************************************************************/
int32_t USER_ReleaseThreadParam( void* pvThreadParam)
{
  OS_Memfree( pvThreadParam);

  return 0;
}

/*****************************************************************************/
/*! Get custom version string for xDriverGetInformation()
*     \param ulMaxLen   Length in Bytes available in szVersion
*     \param szVersion  Buffer for returning string                          */
/*****************************************************************************/
void USER_GetVersionString (uint32_t ulMaxLen, char* szVersion)
{
  (void)OS_Strncpy(szVersion, s_abVer, ulMaxLen);
}


/*****************************************************************************/
/*! Starts the cyclic thread of the netXTransport Toolkit
*   (required for periodic jobs).
*   \param lpParameter  User param (currently not used)
*   \return 0 on success                                                     */
/*****************************************************************************/
static  uint32_t s_fRunCyclicThread = 0;
static DWORD WINAPI CyclicThread( LPVOID lpParameter)
{
  UNREFERENCED_PARAMETER( lpParameter);

  s_fRunCyclicThread = 1;

  while( s_fRunCyclicThread == 1)
  {
    OS_Sleep(100);

    netXTransportCyclicFunction();
  }

  ExitThread(0);
}

/*****************************************************************************/
/*! User defined function to start the toolkit's cyclic thread
*     returns pointer t thread info required to stop (USER_StopCyclicThread) */
/*****************************************************************************/
void* USER_StartCyclicThread(void)
{
  /* start netXTransport cyclic thread */
  HANDLE hThread  = NULL;
  DWORD dwThreadID      = 0;

  hThread = CreateThread( NULL, 0, CyclicThread, NULL, 0, &dwThreadID);

  return hThread;
}

/*****************************************************************************/
/*! User defined function to stop the toolkit's cyclic thread
*     \param pvCyclicThreadInfo User defined parameter                       */
/*****************************************************************************/
void USER_StopCyclicThread( void* pvCyclicThreadInfo)
{
  HANDLE hThread = pvCyclicThreadInfo;

  if (NULL != hThread)
  {
    s_fRunCyclicThread = 0;

    if (WAIT_OBJECT_0 != WaitForSingleObject( hThread, 1000))
      TerminateThread( hThread, 0xFFFFFFFF);

    CloseHandle( hThread);
  }
}
